const { genereteToken } = require("../../config/auth/jwt");
const nodemailer = require("nodemailer");
const path = require("path");
const fs = require("fs");
const bcrypt = require("bcrypt");
const crypto = require("crypto");
const { sendMessage } = require("../../util/gzappy");
const { validarCNPJ } = require("../../util/validarCNPJ");
const { validarCPF } = require("../../util/validarCPF");
const { enviarEmail } = require("../../config/email");
const { default: axios } = require("axios");
const db = require("../../config/db");
const asaas = axios.create({
  baseURL: process.env.ASAAS_BASE_SANDBOX_URL,
  headers: {
    access_token: process.env.ASSAS_API_SANDBOX_KEY,
    "Content-Type": "application/json",
  },
});

class AdminRepository {
  #db = db;
  constructor() {
    this.wsdlUrl = process.env.SOAP_WSDL_URL;
    this.soapAction = process.env.SOAP_ACTION;
    this.S3_BUCKET = process.env.S3_BUCKET;
  }

  /**
   * Registra os dados de um usuario
   * @param {*} data dados do usuario
   * @returns mensagem referente ao resultado obtido
   */
  async register(data) {
    try {
      // 🔎 Validações básicas
      if (!data.nome || !data.documento) {
        return {
          error: true,
          message: "Campos obrigatórios não informados",
          code: 400,
        };
      }

      if (!data.email || !data.celular) {
        return {
          error: true,
          message: "Dados de contato obrigatórios",
          code: 400,
        };
      }

      // 🧼 Normalização
      data.email = data.email.toLowerCase().trim();
      data.celular = data.celular.replace(/\D/g, "");

      const resultado =
        data.documento.length === 11
          ? validarCPF(data.documento)
          : data.documento.length === 14
            ? validarCNPJ(data.documento)
            : null;

      if (!resultado) {
        return {
          error: true,
          message: "CNPJ ou CPF inválido!",
          code: 422,
        };
      }

      const senhaEncripitada = await bcrypt.hash(data.senha, 10);
      const numero = crypto.randomInt(100000, 999999);

      // 💾 Salva usuário no banco

      await this.#db.query(
        `INSERT INTO admins 
        (nome, senha, email, status, celular, role, documento, code)
        VALUES (?,?,?,?,?,?,?,?)`,
        [
          data.nome,
          senhaEncripitada,
          data.email,
          "inativo",
          data.celular,
          data.role,
          data.documento,
          numero,
        ],
      );
      await enviarEmail(data.email, numero, data.documento, data.senha);

      return {
        error: false,
        message: "Admin cadastrado com sucesso",
        code: 201,
      };
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro interno ao cadastrar usuário",
        code: 500,
      };
    }
  }

  /**
   * Realiza o login de um usuario
   * @param {*} data dados do usuario
   * @returns mensagem referente ao resultado obtido
   */
  async login(data) {
    try {
      const admin = await this.#db.query(
        `SELECT * FROM admins WHERE documento = ?`,
        [data.documento],
      );

      if (admin.length === 0) {
        return {
          error: true,
          message: "Usuário não encontrado",
          code: 404,
        };
      }

      const adminDB = admin[0];

      const isMatch = await bcrypt.compare(data.senha, adminDB.senha);

      if (!isMatch) {
        return {
          error: true,
          message: "Senha incorreta",
          code: 401,
        };
      }

      const token = await genereteToken(adminDB);

      return {
        error: false,
        token: token,
        user: {
          nome: adminDB.nome,
          id: adminDB.id,
          documento: adminDB.documento,
          role: adminDB.role,
        },
        code: 200,
      };
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao fazer login",
        code: 500,
      };
    }
  }

  async confirmCode(data) {
    try {
      const code = await this.#db.query(
        `
        SELECT * FROM admins WHERE code = ?
        `,
        [data.code],
      );
      if (code.length === 0) {
        return {
          error: true,
          message: "Código não encontrado",
          code: 404,
        };
      } else {
        await this.#db.query(
          `
          UPDATE admins SET code = null WHERE code = ?
          `,
          [data.code],
        );
        const adminDB = code[0];
        const token = await genereteToken(adminDB);
        return {
          error: false,
          message: "Código confirmado",
          token: token,
          user: {
            nome: adminDB.nome,
            id: adminDB.id,
            documento: adminDB.documento,
            role: adminDB.role,
          },
          code: 200,
        };
      }
    } catch (error) {
      console.error(error);
    }
  }

  async getAdmins() {
    try {
      const admin = await this.#db.query(`
        SELECT 
        id,
        nome,
        documento,
        role,
        status,
        email  
        FROM admins`);
      return admin;
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao buscar administradores",
        code: 500,
      };
    }
  }

  /**
   * Realiza o logout do usuário
   * @param {*} data email do usuário
   * @returns mensagem referente ao resultado obtido
   */
  logout(user) {
    return new Promise(async (resolve, reject) => {
      try {
        this.#db.query(
          "UPDATE usuarios set auth=null WHERE documento=?",
          [user.documento],
          async (error, response) => {
            if (error) {
              console.error(error);
              return reject({
                error: true,
                message: "Erro ao fazer logout",
                code: 500,
              });
            }
            return resolve({
              success: true,
              message: "Logout realizado com sucesso!",
              code: 200,
            });
          },
        );
      } catch (error) {
        console.error(error);
        return reject({
          error: true,
          message: "Erro ao fazer logout",
          code: 500,
        });
      }
    });
  }

  /**
   * Envia um email para recuperação de senha
   * @param {*} data dados do usuario
   * @returns mensagem referente ao resultado obtido
   */
  recover(data) {
    return new Promise(async (resolve, reject) => {
      try {
        this.#db.query(
          "SELECT email FROM usuarios WHERE documento = ?",
          [data.documento],
          async (err, result) => {
            if (err) {
              console.error(err);
              return reject({
                error: true,
                message: `Erro ao buscar usuario`,
                code: 500,
              });
            }

            if (result.length === 0) {
              return reject({
                error: true,
                message: "Usuário não encontrado",
                code: 404,
              });
            }

            let newPassword = Math.floor(100000 + Math.random() * 900000);
            newPassword += "";

            const user = result[0];

            const hashedPassword = await bcrypt.hash(newPassword, 10);

            this.#db.query(
              "UPDATE user SET senha = ? WHERE documento = ?",
              [hashedPassword, data.email],
              (err) => {
                if (err) {
                  console.log(err);
                  return reject({
                    error: true,
                    message: "Erro ao atualizar senha",
                    code: 500,
                  });
                }

                return resolve({
                  success: true,
                  message: "Nova senha enviada para o email!",
                  code: 200,
                });
              },
            );
          },
        );
      } catch (error) {
        console.error(error);
        return reject({
          error: true,
          message: "Erro ao atualizar senha",
          code: 500,
        });
      }
    });
  }

  /**
   * Lista os dados de um usuario
   * @returns lista com todos os dados de um usuario
   */
  getUsuarios(id) {
    return new Promise(async (resolve, reject) => {
      try {
        this.#db.query(
          `SELECT id, nome, documento, role FROM usuarios`,
          [],
          (error, clientes) => {
            if (error) {
              console.error("Erro ao buscar usuario ", error);
              return reject({
                error: true,
                message: `Erro ao buscar usuarios`,
                code: 500,
              });
            }
            return resolve(clientes);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar usuarios ", error);
        return reject({
          error: true,
          message: `Erro ao buscar usuarios`,
          code: 500,
        });
      }
    });
  }

  getUmUsuario(id) {
    return new Promise(async (resolve, reject) => {
      try {
        this.#db.query(
          `SELECT id, nome, documento, role FROM usuarios WHERE id = ?`,
          [id],
          (error, usuario) => {
            if (error) {
              console.error("Erro ao buscar usuario ", error);
              return reject({
                error: true,
                message: `Erro ao buscar usuario`,
                code: 500,
              });
            }

            if (usuario.length === 0) {
              return reject({
                error: true,
                message: `Nenhum usuário com esse id`,
                code: 404,
              });
            }

            return resolve(usuario[0]);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar usuário ", error);
        return reject({
          error: true,
          message: `Erro ao buscar usuário`,
          code: 500,
        });
      }
    });
  }

  /**
   * Deleta o cliente cadastrado
   * @param {*} id ID do cliente a ser excluído
   * @returns mensagem referente ao resultado obtido
   */
  delete(id) {
    return new Promise((resolve, reject) => {
      console.log(id);
      this.#db.query(
        "DELETE FROM usuarios WHERE id = ?",
        [id],
        (error, result) => {
          console.log("RESULTADO DELETE:", result);
          console.log("ERRO DELETE:", error);

          if (error) {
            console.error("Erro ao deletar usuario: ", error);
            return resolve({
              error: true,
              message: "Este usuário possui vínculos e não pode ser deletado!",
              code: 409,
            });
          }

          if (result.affectedRows === 0) {
            return resolve({
              error: true,
              message: "Usuário não encontrado",
              code: 404,
            });
          }

          return resolve({
            success: true,
            message: "Usuário deletado com sucesso",
            code: 200,
          });
        },
      );
    });
  }

  /**
   * Atualiza os dados de um usuario
   * @param {*} data dados atualizados do usuario
   * @returns mensagem referente ao resultado obtido
   */
  update(data) {
    return new Promise(async (resolve, reject) => {
      try {
        this.#db.query(
          "UPDATE usuarios SET nome=?, email=?, role = ?  WHERE id=?",
          [data.nome, data.email, data.role, data.id],
          async (error, response) => {
            if (error) {
              console.error(`Erro ao atualizar usuario: `, error);
              return reject({
                error: true,
                message: `Erro ao atualizar usuário`,
                code: 500,
              });
            }
            return resolve({
              success: true,
              message: "Usuário atualizado com sucesso!",
              code: 200,
            });
          },
        );
      } catch (error) {
        console.error(`Erro ao atualizar usuario: `, error);
        return reject({
          error: true,
          message: `Erro ao atualizar usuário`,
          code: 500,
        });
      }
    });
  }

  /**
   * Verifica o código enviado por email e valida
   * @param {*} code código recebido no email
   * @returns mensagem referente ao resultado obtido
   */
  verify(code) {
    return new Promise((resolve, reject) => {
      this.#db.query(
        "SELECT * FROM usuarios WHERE codigo=?",
        [code],
        async (error, response) => {
          if (error) return reject(new Error(error));
          try {
            if (response[0]) {
              this.#db.query(
                "UPDATE usuarios SET codigo=true WHERE codigo=?",
                [code],
                (error, response) => {
                  if (error) return reject(new Error(error));
                  return resolve({ success: `Código verificado!` });
                },
              );
            } else
              return resolve({
                error: "Não foi possível verificar!",
                code: 203,
              });
          } catch (error) {
            return reject(new Error(error));
          }
        },
      );
    });
  }

  /**
   * Criptografa um valor
   * @param {*} data valor recebido
   * @returns o valor criptografado
   */
  encrypt(data) {
    return new Promise(async (resolve, reject) => {
      try {
        let value;
        if (data.hash) {
          value = await generateHash(data.value);
        } else {
          value = await encrypt(data.value);
        }
        return resolve({ value: value });
      } catch (error) {
        return resolve({ error: `Não foi possível criptografar!`, code: 203 });
      }
    });
  }

  ping() {
    return new Promise(async (resolve, reject) => {
      const dateAtual = new Date();
      const horaFormatada = dateAtual.toLocaleTimeString("pt-BR", {
        hour: "2-digit",
        minute: "2-digit",
      });

      const dia = String(dateAtual.getDate()).padStart(2, "0");
      const mes = String(dateAtual.getMonth() + 1).padStart(2, "0");
      const ano = dateAtual.getFullYear();
      const dataFormatada = `${dia}/${mes}/${ano}`;

      // Concatena data e hora
      const dataHoraFormatada = `${dataFormatada} ${horaFormatada}`;

      return resolve({
        status: 200,
        api: `Croissant Pedidos`,
        data: dataHoraFormatada,
      });
    });
  }
}

module.exports = new AdminRepository();
