const { default: axios } = require("axios");
const db = require("../../config/db");
const { sendMessage } = require("../../util/gzappy");
const asaas = axios.create({
  baseURL: process.env.ASAAS_BASE_URL,
  headers: {
    access_token: process.env.ASAAS_API_KEY,
    "Content-Type": "application/json",
  },
});

class BoletosRepository {
  #db = db;

  /* ================= CREATE ================= */
  async create(data) {
    try {
      const getCusId = await this.#db.query(
        `SELECT idAsaas, celular FROM clientes WHERE documento =?`,
        [data.documento],
      );

      const cusId = getCusId[0].idAsaas;
      const celular = getCusId[0].celular;

      const gerarBoleto = await this.gerarBoleto(data, cusId);

      if (gerarBoleto.error) {
        return {
          error: true,
          message: "Erro ao gerar boleto",
          code: 500,
          detail: gerarBoleto.detail,
        };
      } else {
        await this.#db.query(
          `INSERT INTO boletos (documento, data_criado, data_vencimento, natureza, ref_interna, descricao, preco_geral, boleto_status_asaas, paymentId)
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
          [
            data.documento,
            data.data_criado,
            data.data_vencimento,
            data.natureza,
            data.ref_interna || null,
            data.descricao,
            data.preco_geral,
            gerarBoleto.data.status,
            gerarBoleto.data.id,
          ],
        );

        await sendMessage(
          `Boleto criado com sucesso para o documento ${data.documento}\n\n Link do para pagar o boleto: ${gerarBoleto.data.bankSlipUrl}`,
          "55" + celular,
        );

        return {
          error: false,
          message: "Boleto criado com sucesso!",
          code: 201,
        };
      }
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao criar boleto",
        code: 500,
        detail: error,
      };
    }
  }
  /* ================= CRIAR ASSINATURA ================= */
  async createAssinatura(data) {
    try {
      const getCusId = await this.#db.query(
        `SELECT idAsaas FROM usuarios WHERE documento =?`,
        [data.documento],
      );
      const cusId = getCusId[0].idAsaas;

      const gerarAssinatura = await this.gerarAssinatura(data, cusId);
      if (gerarAssinatura.error) {
        return {
          error: true,
          message: "Erro ao gerar assinatura",
          code: 500,
          detail: gerarAssinatura.detail,
        };
      } else {
        this.#db.query(
          `INSERT INTO assinatura
            (documento, data_criado, data_vencimento, identificacao, descricao,
            preco_geral, boleto_status_asaas, paymentId, nota_fiscal_asaas,
            frequencia, forma_pagamento)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
          [
            data.documento,
            data.data_criado,
            data.data_vencimento,
            data.natureza,
            data.descricao,
            data.preco_geral,
            gerarAssinatura.data.status,
            gerarAssinatura.data.id,
            null,
            data.ciclo || "MONTHLY",
            data.tipoPagamento,
          ],
        );

        return {
          error: false,
          message: "Assinatura criada com sucesso!",
          code: 201,
        };
      }
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao criar assinatura",
        code: 500,
        detail: error,
      };
    }
  }
  /* ================= TRASNFERIR ================= */
  async transferir(data) {
    try {
      const mappedData = {
        value: data.valor,
        pixKey: data.pixKey,
      };

      switch (data.pixKeyType) {
        case pixAddressKeyType.length === 11:
          mappedData.pixKeyType = "CPF";
          break;
        case pixAddressKeyType.length === 14:
          mappedData.pixKeyType = "CNPJ";
          break;
        case pixAddressKeyType.includes("@"):
          mappedData.pixKeyType = "EMAIL";
          break;
        case pixAddressKeyType.match(/^\d{4}-\d{4}-\d{4}-\d{4}$/):
          mappedData.pixKeyType = "PHONE";
          break;
        default:
          return {
            error: true,
            message: "Tipo de chave PIX inválida",
            code: 422,
          };
      }

      const response = await this.transferirAsaas(mappedData);
      return response;
    } catch (error) {
      console.error("Erro ao mapear tipo de chave PIX:", error);
      return {
        error: true,
        message: "Erro ao mapear tipo de chave PIX",
        code: 500,
        detail: error.message,
      };
    }
  }

  /* ================= LISTAR TODOS ================= */
  async getBoletos() {
    try {
      const boletos = await this.#db.query(`SELECT * FROM boletos`);

      return boletos;
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao buscar boletos",
        code: 500,
        detail: error,
      };
    }
  }

  async getQtdeBoletosNaoPagos() {
    try {
      const boletos = await this.#db.query(
        `SELECT COUNT(*) AS boletos_pendentes FROM boletos WHERE boleto_status_asaas = "PENDING"`,
      );

      return boletos[0];
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao buscar boletos",
        code: 500,
        detail: error,
      };
    }
  }

  async getBoletosPendentes() {
    try {
      const boletos = await this.#db.query(`
        SELECT 
          DATE_FORMAT(boletos.data_vencimento, '%d/%m/%Y') AS data_vencimento,
          usuarios.nome,
          GREATEST(DATEDIFF(CURDATE(), boletos.data_vencimento), 0) AS dias_em_atraso,
          boletos.preco_geral,
          (
              SELECT COUNT(*)
              FROM boletos b2
              WHERE b2.boleto_status_asaas = 'PENDING'
          ) AS boletos_pendentes
      FROM boletos
      JOIN usuarios ON boletos.id_user = usuarios.id;`);

      return boletos;
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao buscar boletos",
        code: 500,
        detail: error,
      };
    }
  }

  async setBoletosStatus(paymentId) {
    try {
      await this.#db.query(
        `UPDATE boletos set boleto_status_asaas = "PAYMENT_RECEIVED" WHERE paymentId = ?`,
        [paymentId],
      );

      return {
        error: false,
        message: "Status do boleto atualizado com sucesso!",
        code: 200,
      };
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao atualizar status do boleto",
        code: 500,
        detail: error,
      };
    }
  }

  /* ================= BUSCAR UM ================= */
  async getUmBoleto(id) {
    try {
      const boletos = await this.#db.query(
        `SELECT *
         FROM boletos
         WHERE id = ?`,
        [id],
      );

      if (!boletos.length) {
        return {
          error: true,
          message: "Boleto não encontrado",
          code: 404,
        };
      }

      return boletos[0];
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao buscar boleto",
        code: 500,
        detail: error,
      };
    }
  }

  async getBoletosAtrasados() {
    try {
      const boletos = await this.#db.query(
        `SELECT *
       FROM boletos
       WHERE data_vencimento < CURDATE()
       AND boleto_status_asaas = 'PENDING'`,
      );

      return boletos;
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao buscar boletos atrasados",
        code: 500,
        detail: error,
      };
    }
  }

  /* ================= UPDATE ================= */
  async update(data) {
    try {
      const fields = [];
      const values = [];

      if (data.documento) {
        fields.push("documento = ?");
        values.push(data.documento);
      }

      if (data.data_criado) {
        fields.push("data_criado = ?");
        values.push(data.data_criado);
      }

      if (!fields.length) {
        return {
          error: true,
          message: "Nenhum dado para atualizar",
          code: 422,
        };
      }

      values.push(data.id);

      const query = `
        UPDATE boletos
        SET ${fields.join(", ")}
        WHERE id = ?
      `;

      const result = await this.#db.query(query, values);

      if (result.affectedRows === 0) {
        return {
          error: true,
          message: "Boleto não encontrado",
          code: 404,
        };
      }

      return {
        error: false,
        message: "Boleto atualizado com sucesso!",
        code: 200,
      };
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao atualizar boleto",
        code: 500,
        detail: error,
      };
    }
  }

  /* ================= DELETE ================= */
  async delete(id) {
    try {
      const result = await this.#db.query(`DELETE FROM boletos WHERE id = ?`, [
        id,
      ]);

      if (result.affectedRows === 0) {
        return {
          error: true,
          message: "Boleto não encontrado",
          code: 404,
        };
      }

      return {
        error: false,
        message: "Boleto deletado com sucesso!",
        code: 200,
      };
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao deletar boleto",
        code: 500,
        detail: error,
      };
    }
  }

  async getOrCreateCustomer(cliente) {
    const cpfCnpj = cliente.cpfCnpj.replace(/\D/g, "");

    try {
      const busca = await asaas.get("/customers", {
        params: { cpfCnpj },
      });

      if (busca.data.data.length > 0) {
        console.log("COSTUMER NA ASAAS: ", busca.data.data[0]);
        return busca.data.data[0].id;
      }

      const criado = await asaas.post("/customers", {
        name: cliente.nome,
        cpfCnpj,
        email: cliente.email,
      });

      return criado.data.id;
    } catch (err) {
      const retry = await asaas.get("/customers", {
        params: { cpfCnpj },
      });

      if (retry.data.data.length > 0) {
        return retry.data.data[0].id;
      }

      throw err;
    }
  }

  async gerarBoleto(data, customerId) {
    try {
      // 1. Busca ou cria o cliente
      //const customerId = await this.getOrCreateCustomer(data.cliente);

      // 2. Gera o boleto
      const response = await axios.post(
        `${process.env.ASAAS_BASE_SANDBOX_URL}/payments`,
        {
          customer: customerId,
          billingType: "BOLETO",
          value: data.preco_geral,
          dueDate: data.data_vencimento,
          description: data.descricao,
        },
        {
          headers: {
            access_token: process.env.ASSAS_API_SANDBOX_KEY,
          },
        },
      );

      return { data: response.data, code: 200 };
    } catch (error) {
      console.error(
        "Erro ao gerar boleto:",
        error.response?.data || error.message,
      );
      return {
        error: true,
        message: "Erro ao gerar boleto",
        code: 500,
        detail: error.response?.data || error.message,
      };
    }
  }

  async gerarAssinatura(data, customerId) {
    try {
      const response = await axios.post(
        `${process.env.ASAAS_BASE_SANDBOX_URL}/subscriptions`,
        {
          customer: customerId,
          billingType: data.tipoPagamento,
          value: data.preco_geral,
          nextDueDate: data.data_vencimento,
          cycle: data.ciclo || "MONTHLY",
          description: data.descricao,
        },
        {
          headers: {
            access_token: process.env.ASSAS_API_SANDBOX_KEY,
          },
        },
      );
      return { data: response.data, code: 200 };
    } catch (error) {
      console.error(
        "Erro ao gerar assinatura:",
        error.response?.data || error.message,
      );
      return {
        error: true,
        message: "Erro ao gerar assinatura",
        code: 500,
        detail: error.response?.data || error.message,
      };
    }
  }

  async getTotalConta() {
    try {
      const response = await axios.get(
        `${process.env.ASAAS_BASE_SANDBOX_URL}/finance/balance`,
        {
          headers: {
            access_token: process.env.ASSAS_API_SANDBOX_KEY,
            Accept: "application/json",
          },
        },
      );
      return { data: response.data, code: 200 };
    } catch (error) {
      console.error(
        "Erro ao pegar total da conta",
        error.response?.data || error.message,
      );
      return {
        error: true,
        message: "Erro ao pegar total da conta",
        code: 500,
        detail: error.response?.data || error.message,
      };
    }
  }

  async transferirAsaas(data) {
    try {
      const response = await axios.post(
        `${process.env.ASAAS_BASE_SANDBOX_URL}/transfers`,
        {
          value: data.valor,
          pixAddressKey: data.pixKey,
          pixAddressKeyType: data.pixKeyType,
          operationType: "PIX",
        },
        {
          headers: {
            access_token: process.env.ASSAS_API_SANDBOX_KEY,
          },
        },
      );
      return { data: response.data, code: 200 };
    } catch (error) {
      console.error(
        "Erro ao transferir:",
        error.response?.data || error.message,
      );
      return {
        error: true,
        message: "Erro ao transferir",
        code: 500,
        detail: error.response?.data || error.message,
      };
    }
  }

  // async gerarNotaFiscal(paymentId, dadosNota) {
  //   try {
  //     const response = await asaas.post("/invoices", {
  //       payment: paymentId,
  //       serviceDescription: dadosNota.serviceDescription,
  //       value: dadosNota.value,
  //       deductions: 0,
  //       taxes: {
  //         iss: dadosNota.iss ?? 0
  //       }
  //     });

  //     return response.data;

  //   } catch (error) {
  //     console.error(
  //       "Erro ao gerar nota fiscal:",
  //       error.response?.data || error.message
  //     );

  //     return {
  //       error: true,
  //       message: "Erro ao gerar nota fiscal",
  //       detail: error.response?.data || error.message
  //     };
  //   }
  // }

  async gerarNotaFiscal(dadosNota) {
    try {
      console.log("GERANDO NOTA FISCAL DADOS NOTA ", dadosNota);

      const response = await asaas.post("/invoices", {
        taxes: {
          inss: 0,
          retainIss: false,
          iss: 0,
          ir: 0,
          pis: 0,
          csll: 0,
          cofins: 0,
        },
        serviceDescription: "Desenvolvimento de Sistemas",
        municipalServiceCode: "01.01.01",
        value: 5,
        customerProduct: {
          municipalServiceExternalId: null,
          municipalServiceCode: "01.01.01",
          name: "01.01.01",
          issTax: 0,
        },
        effectiveDate: dadosNota.effectiveDate,
        customer: dadosNota.customer,
      });

      return response.data;
    } catch (error) {
      console.error(
        "Erro ao gerar nota fiscal:",
        error.response?.data || error.message,
      );

      return {
        error: true,
        message: "Erro ao gerar nota fiscal",
        detail: error.response?.data || error.message,
      };
    }
  }

  async gerarBoletoComNota(data) {
    try {
      // 1. Gera o boleto
      const boleto = await this.gerarBoleto(data);

      if (boleto.error) return boleto;

      // 2. Gera a nota fiscal
      // const notaFiscal = await this.gerarNotaFiscal(boleto.id, {
      //   descricao: "Mensalidade do sistema",
      //   valor: data.valor,
      //   iss: 5
      // });

      return {
        boleto,
      };
    } catch (error) {
      console.error("Erro ao gerar boleto com nota:", error);
      return {
        error: true,
        message: "Erro ao gerar boleto com nota",
        code: 500,
        detail: error,
      };
    }
  }
}

module.exports = new BoletosRepository();
