const { genereteToken } = require("../../config/auth/jwt");
const nodemailer = require("nodemailer");
const path = require("path");
const fs = require("fs");
const bcrypt = require("bcrypt");
const crypto = require("crypto");
const { sendMessage } = require("../../util/gzappy");
const { validarCNPJ } = require("../../util/validarCNPJ");
const { validarCPF } = require("../../util/validarCPF");

const { default: axios } = require("axios");
const db = require("../../config/db");
const asaas = axios.create({
  baseURL: process.env.ASAAS_BASE_SANDBOX_URL,
  headers: {
    access_token: process.env.ASSAS_API_SANDBOX_KEY,
    "Content-Type": "application/json",
  },
});

class ClientesRepository {
  #db = db;
  constructor() {
    this.wsdlUrl = process.env.SOAP_WSDL_URL;
    this.soapAction = process.env.SOAP_ACTION;
    this.S3_BUCKET = process.env.S3_BUCKET;
  }

  /**
   * Registra os dados de um usuario
   * @param {*} data dados do usuario
   * @returns mensagem referente ao resultado obtido
   */
  async register(data) {
    try {
      // 🔎 Validações básicas
      if (!data.nome || !data.documento) {
        return {
          error: true,
          message: "Campos obrigatórios não informados",
          code: 400,
        };
      }

      if (!data.email || !data.celular || !data.cep) {
        return {
          error: true,
          message: "Dados de contato obrigatórios",
          code: 400,
        };
      }

      // 🧼 Normalização
      data.email = data.email.toLowerCase().trim();
      data.celular = data.celular.replace(/\D/g, "");
      data.cep = data.cep.replace(/\D/g, "");
      data.documento = data.documento.replace(/\D/g, "");

      // 🧾 Validação CPF/CNPJ
      const resultado =
        data.documento.length === 11
          ? validarCPF(data.documento)
          : data.documento.length === 14
            ? validarCNPJ(data.documento)
            : null;

      if (!resultado) {
        return {
          error: true,
          message: "CNPJ ou CPF inválido!",
          code: 422,
        };
      }

      // 🟦 Payload Asaas
      const criarAsaas = {
        name: data.nome,
        cpfCnpj: data.documento,
        email: data.email,
        mobilePhone: data.celular,
        address: data.logradouro,
        addressNumber: data.numero,
        province: data.bairro,
        postalCode: data.cep,
      };

      // 🚀 Cria customer no Asaas
      const response = await asaas.post("/customers", criarAsaas);

      if (!response?.data?.id) {
        return {
          error: true,
          message: "Erro ao criar cliente no Asaas",
          code: 502,
        };
      }

      const idAsaas = response.data.id;

      // 💾 Salva usuário no banco
      try {
        await this.#db.query(
          `INSERT INTO clientes 
        (nome, role, documento, tipo, email, celular, cep, logradouro, numero, bairro, cidade, uf, idAsaas, whatsapp, enEmail, status, senha)
        VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)`,
          [
            data.nome,
            "USER",
            data.documento,
            data.tipo,
            data.email,
            data.celular,
            data.cep,
            data.logradouro,
            data.numero,
            data.bairro,
            data.cidade,
            data.uf,
            idAsaas,
            data.enWhatsapp,
            data.enEmail,
            "inativo",
            data.senha ? await bcrypt.hash(data.senha, 10) : null,
          ],
        );
      } catch (dbError) {
        // 🔄 Rollback Asaas se DB falhar
        await asaas.delete(`/customers/${idAsaas}`);
        throw dbError;
      }

      // ✅ Retorno final
      return {
        error: false,
        message: "Usuário cadastrado com sucesso",
        idAsaas,
        code: 201,
      };
    } catch (error) {
      if (error.response) {
        console.error("Erro Asaas:", error.response.data);
      } else {
        console.error(error);
      }

      return {
        error: true,
        message: "Erro interno ao cadastrar usuário",
        code: 500,
      };
    }
  }

  async getUser() {
    try {
      const user = await this.#db.query(`
        SELECT * FROM clientes`);
      return user;
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao buscar usuário",
        code: 500,
        detail: error,
      };
    }
  }

  async getUserComOs() {
    try {
      const user = await this.#db.query(`
        SELECT u.id, u.nome, u.documento, u.email, u.celular, u.cep, u.logradouro, u.numero, u.bairro, u.cidade, u.uf,
        (SELECT COUNT(*) FROM boletos os WHERE os.documento = u.documento) AS total_os,
        (SELECT SUM(preco_geral) FROM boletos os WHERE os.documento = u.documento) AS total_valor
        FROM clientes u`);
      return user;
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao buscar clientes",
        code: 500,
        detail: error,
      };
    }
  }

  async getUsersInadimplentes() {
    try {
      const today = new Date().toISOString().split("T")[0];
      const users = await this.#db.query(`
       SELECT b.id AS boleto_id, u.nome, u.documento, u.email, u.celular,  b.data_vencimento, b.preco_geral
        FROM clientes u
        JOIN boletos b ON b.documento = u.documento
        WHERE b.boleto_status_asaas = 'PENDING' AND b.data_vencimento < '${today}'
        `);
      return users;
    } catch (error) {
      console.error(error);
      throw error;
    }
  }

  async getUsersFluxoFiscal() {
    try {
      const today = new Date().toISOString().split("T")[0];
      const users = await this.#db.query(`
       SELECT b.id AS boleto_id, u.nome, u.documento, u.email, u.celular,  b.data_vencimento, b.boleto_status_asaas, b.preco_geral
        FROM clientes u
        JOIN boletos b ON b.documento = u.documento
        WHERE b.data_vencimento > '${today}'
        `);
      return users;
    } catch (error) {
      console.error(error);
      throw error;
    }
  }

  /**
   * Lista os dados de um usuario
   * @returns lista com todos os dados de um usuario
   */
  getclientes(id) {
    return new Promise(async (resolve, reject) => {
      try {
        this.#db.query(
          `SELECT id, nome, documento, role FROM clientes`,
          [],
          (error, clientes) => {
            if (error) {
              console.error("Erro ao buscar usuario ", error);
              return reject({
                error: true,
                message: `Erro ao buscar clientes`,
                code: 500,
              });
            }
            return resolve(clientes);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar clientes ", error);
        return reject({
          error: true,
          message: `Erro ao buscar clientes`,
          code: 500,
        });
      }
    });
  }

  getUmCliente(id) {
    return new Promise(async (resolve, reject) => {
      try {
        this.#db.query(
          `SELECT * FROM clientes WHERE id = ?`,
          [id],
          (error, usuario) => {
            if (error) {
              console.error("Erro ao buscar cliente ", error);
              return reject({
                error: true,
                message: `Erro ao buscar cliente`,
                code: 500,
              });
            }

            if (usuario.length === 0) {
              return reject({
                error: true,
                message: `Nenhum cliente com esse id`,
                code: 404,
              });
            }

            return resolve(usuario[0]);
          },
        );
      } catch (error) {
        console.error("Erro ao buscar usuário ", error);
        return reject({
          error: true,
          message: `Erro ao buscar usuário`,
          code: 500,
        });
      }
    });
  }

  /**
   * Deleta o cliente cadastrado
   * @param {*} id ID do cliente a ser excluído
   * @returns mensagem referente ao resultado obtido
   */
  delete(id) {
    return new Promise((resolve, reject) => {
      console.log(id);
      this.#db.query(
        "DELETE FROM clientes WHERE id = ?",
        [id],
        (error, result) => {
          console.log("RESULTADO DELETE:", result);
          console.log("ERRO DELETE:", error);

          if (error) {
            console.error("Erro ao deletar usuario: ", error);
            return resolve({
              error: true,
              message: "Este cliente possui vínculos e não pode ser deletado!",
              code: 409,
            });
          }

          if (result.affectedRows === 0) {
            return resolve({
              error: true,
              message: "Cliente não encontrado",
              code: 404,
            });
          }

          return resolve({
            error: false,
            message: "Cliente deletado com sucesso",
            code: 200,
          });
        },
      );
    });
  }

  /**
   * Atualiza os dados de um usuario
   * @param {*} data dados atualizados do usuario
   * @returns mensagem referente ao resultado obtido
   */
  update(data) {
    return new Promise(async (resolve, reject) => {
      try {
        this.#db.query(
          "UPDATE clientes SET nome=?, email=?, role = ?  WHERE id=?",
          [data.nome, data.email, data.role, data.id],
          async (error, response) => {
            if (error) {
              console.error(`Erro ao atualizar cliente: `, error);
              return reject({
                error: true,
                message: `Erro ao atualizar cliente`,
                code: 500,
              });
            }
            return resolve({
              error: false,
              message: "Cliente atualizado com sucesso!",
              code: 200,
            });
          },
        );
      } catch (error) {
        console.error(`Erro ao atualizar cliente: `, error);
        return reject({
          error: true,
          message: `Erro ao atualizar cliente`,
          code: 500,
        });
      }
    });
  }

  /**
   * Verifica o código enviado por email e valida
   * @param {*} code código recebido no email
   * @returns mensagem referente ao resultado obtido
   */
  verify(code) {
    return new Promise((resolve, reject) => {
      this.#db.query(
        "SELECT * FROM clientes WHERE codigo=?",
        [code],
        async (error, response) => {
          if (error) return reject(new Error(error));
          try {
            if (response[0]) {
              this.#db.query(
                "UPDATE clientes SET codigo=true WHERE codigo=?",
                [code],
                (error, response) => {
                  if (error) return reject(new Error(error));
                  return resolve({ success: `Código verificado!` });
                },
              );
            } else
              return resolve({
                error: "Não foi possível verificar!",
                code: 203,
              });
          } catch (error) {
            return reject(new Error(error));
          }
        },
      );
    });
  }

  /**
   * Criptografa um valor
   * @param {*} data valor recebido
   * @returns o valor criptografado
   */
  encrypt(data) {
    return new Promise(async (resolve, reject) => {
      try {
        let value;
        if (data.hash) {
          value = await generateHash(data.value);
        } else {
          value = await encrypt(data.value);
        }
        return resolve({ value: value });
      } catch (error) {
        return resolve({ error: `Não foi possível criptografar!`, code: 203 });
      }
    });
  }

  ping() {
    return new Promise(async (resolve, reject) => {
      const dateAtual = new Date();
      const horaFormatada = dateAtual.toLocaleTimeString("pt-BR", {
        hour: "2-digit",
        minute: "2-digit",
      });

      const dia = String(dateAtual.getDate()).padStart(2, "0");
      const mes = String(dateAtual.getMonth() + 1).padStart(2, "0");
      const ano = dateAtual.getFullYear();
      const dataFormatada = `${dia}/${mes}/${ano}`;

      // Concatena data e hora
      const dataHoraFormatada = `${dataFormatada} ${horaFormatada}`;

      return resolve({
        status: 200,
        api: `Croissant Pedidos`,
        data: dataHoraFormatada,
      });
    });
  }
}

module.exports = new ClientesRepository();
