const bcrypt = require("bcrypt");
const { genereteToken } = require("../../config/auth/jwt");
const db = require("../../config/db");

class UsuariosRepository {
  #db = db;

  /* ================= REGISTER ================= */
  async register(data) {
    try {
      const hashedSenha = await bcrypt.hash(data.senha, 10);

      await this.#db.query(
        `INSERT INTO usuarios 
        (nome, role, documento, email, celular, senha)
        VALUES (?,?,?,?,?,?)`,
        [
          data.nome,
          "USER",
          data.documento,
          data.email,
          data.celular,
          hashedSenha,
        ]
      );

      return {
        error: false,
        message: "Usuário cadastrado com sucesso",
        code: 201,
      };
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao cadastrar usuário",
        code: 500,
        detail: error,
      };
    }
  }

  /* ================= LOGIN ================= */
  async login(data) {
    try {
      const response = await this.#db.query(
        `SELECT * FROM usuarios WHERE email = ?`,
        [data.email]
      );

      if (!response.length) {
        return {
          error: true,
          message: "Usuário não encontrado",
          code: 404,
        };
      }

      const userDB = response[0];

      const isMatch = await bcrypt.compare(data.senha, userDB.senha);

      if (!isMatch) {
        return {
          error: true,
          message: "Senha incorreta",
          code: 401,
        };
      }

      const token = await genereteToken(userDB);

      return {
        token,
        user: {
          id: userDB.id,
          nome: userDB.nome,
          documento: userDB.documento,
          role: userDB.role,
        },
        code: 200,
      };
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao fazer login",
        code: 500,
        detail: error,
      };
    }
  }

  /* ================= LISTAR TODOS ================= */
  async getUsuarios() {
    try {
      const usuarios = await this.#db.query(
        `SELECT id, nome, email, documento, role FROM usuarios`
      );
      return usuarios;
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao buscar usuários",
        code: 500,
      };
    }
  }

  /* ================= BUSCAR UM ================= */
  async getUmUsuario(id) {
    try {
      const usuario = await this.#db.query(
        `SELECT id, nome, email, role, documento FROM usuarios WHERE id = ?`,
        [id]
      );

      if (!usuario.length) {
        return {
          error: true,
          message: "Usuário não encontrado",
          code: 404,
        };
      }

      return usuario[0];
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao buscar usuário",
        code: 500,
      };
    }
  }

  /* ================= UPDATE ================= */
  async update(data) {
    try {
      const result = await this.#db.query(
        `UPDATE usuarios SET nome=?, email=?, role=?, documento=? WHERE id=?`,
        [data.nome, data.email, data.role, data.documento, data.id]
      );

      if (result.affectedRows === 0) {
        return {
          error: true,
          message: "Usuário não encontrado",
          code: 404,
        };
      }

      return {
        error: false,
        message: "Usuário atualizado com sucesso",
        code: 200,
      };
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao atualizar usuário",
        code: 500,
      };
    }
  }

  /* ================= DELETE ================= */
  async delete(id) {
    try {
      const result = await this.#db.query(
        `DELETE FROM usuarios WHERE id = ?`,
        [id]
      );

      if (result.affectedRows === 0) {
        return {
          error: true,
          message: "Usuário não encontrado",
          code: 404,
        };
      }

      return {
        error: false,
        message: "Usuário deletado com sucesso",
        code: 200,
      };
    } catch (error) {
      console.error(error);
      return {
        error: true,
        message: "Erro ao deletar usuário",
        code: 500,
      };
    }
  }
}

module.exports = new UsuariosRepository();
